//
//  ItemRow.swift
//  Do It
//
//  Created by Jim Dovey on 4/13/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI



struct ItemRow: View {
    @EnvironmentObject var data: DataCenter
    @State var item: TodoItem
    
    @State var editingTitle: Bool = false
    @State var hovering: Bool = false
    @State var showInfo: Bool = false
    
    var body: some View {
        HStack(alignment: .firstTextBaseline) {
            Button(action: {
                self.item.complete.toggle()
                self.data.updateTodoItem(self.item)
            }) {
                Image(item.complete
                        ? "largecircle.fill.circle"
                        : "circle")
                    .renderingMode(.template)
                    .resizable()
                    .frame(width: 22, height: 22)
                    .foregroundColor(.accentColor)
                    .alignmentGuide(.firstTextBaseline) { $0[.bottom] * 0.7 }
            }
            .padding(.trailing, 6)
            .buttonStyle(HighPriorityButtonStyle())
            
            VStack(alignment: .leading) {
                HStack {
                    TextField("Title", text: $item.title,
                              onEditingChanged: { self.editingTitle = $0 },
                              onCommit: { self.data.updateTodoItem(self.item) })
                        .font(.headline)
                        .foregroundColor(item.complete ? .secondary : .primary)
                        .textFieldStyle(PlainTextFieldStyle())
                    
                    Spacer()
                    
                    PriorityBadge(priority: item.priority)
                }
                
                if item.notes != nil && !item.notes!.isEmpty {
                    Text(item.notes!)
                        .foregroundColor(.secondary)
                        .lineLimit(nil)
                }
                
                if item.date != nil {
                    Text("\(item.date!, formatter: Formatters.optionalTime(in: item.date!))")
                        .font(.caption)
                        .foregroundColor(item.date! < Date() ? .red : .secondary)
                }
            }
            
            if editingTitle || hovering {
                Button(action: { self.showInfo.toggle() }) {
                    Image("info.circle")
                        .resizable()
                        .renderingMode(.template)
                        .frame(width: 20, height: 20)
                        .alignmentGuide(.firstTextBaseline) { $0[.bottom] * 0.76 }
                        .foregroundColor(.accentColor)
                }
                .buttonStyle(PlainButtonStyle())
            }
        }
        .onDrag { self.data.itemProvider(forItem: self.item) }
        .onHover { self.hovering = $0 }
    }
}

struct ItemRow_Previews: PreviewProvider {
    static var previews: some View {
        return ForEach(0..<6) {
            ItemRow(item: defaultTodoItems[$0])
                .padding()
        }
        .environmentObject(DataCenter())
    }
}
